"""Repository for Task entities."""

from __future__ import annotations

from datetime import date
from typing import List, Optional
from tinydb import Query

from ca_task_manager.models.task import Task
from ca_task_manager.repositories.base import TinyDBRepository


class TaskRepository(TinyDBRepository):
    def __init__(self, db_path: str = "data/tasks.json"):
        super().__init__(db_path, table_name="tasks", model_class=Task)

    def create_task(self, task: Task) -> int:
        """Insert a new task after ensuring unique task_code and normalising dates."""
        existing = self.table.get(Query().task_code == task.task_code)
        if existing:
            raise ValueError(f"Task code {task.task_code} already exists.")
        data = task.dict()
        # Normalise due_date to ISO string for JSON persistence
        due = data.get("due_date")
        if hasattr(due, "isoformat"):
            data["due_date"] = due.isoformat()
        return self.insert(data)

    def update_task(self, task_id: int, updates: dict) -> bool:
        # Check if task_code is being updated
        if "task_code" in updates:
            existing = self.table.get((Query().task_code == updates["task_code"]) & (Query().id != task_id))
            if existing:
                raise ValueError(f"Task code {updates['task_code']} already exists.")
        # Convert date to iso string
        if "due_date" in updates:
            val = updates["due_date"]
            if hasattr(val, "isoformat"):
                updates["due_date"] = val.isoformat()
        return self.update(task_id, updates)

    def find_by_task_code(self, code: str) -> Optional[dict]:
        return self.table.get(Query().task_code == code)

    def tasks_due_within(self, start_date: date, end_date: date) -> List[dict]:
        """Return tasks whose due_date falls between start_date and end_date (inclusive).

        TinyDB stores dates as ISO strings.  We filter in Python to support comparisons.
        """
        results: List[dict] = []
        for item in self.table.all():
            due_val = item.get("due_date")
            if not due_val:
                continue
            # Convert to date
            if isinstance(due_val, str):
                try:
                    due_dt = date.fromisoformat(due_val)
                except ValueError:
                    continue
            elif isinstance(due_val, date):
                due_dt = due_val
            else:
                continue
            if start_date <= due_dt <= end_date:
                results.append(item)
        return results